#!/bin/bash
set -e

# Test gosec XML output on a simple hello world app
VERSIONS=(
  "v2.22.0"
  "v2.20.0"
  "v2.18.0"
  "v2.15.0"
)

# Create test directory with a simple Go app
TEST_DIR="gosec-helloworld-$(date +%s)"
mkdir -p "$TEST_DIR"

# Create a simple Go module with intentional security issues
cat > "$TEST_DIR/go.mod" <<'EOF'
module helloworld

go 1.22
EOF

cat > "$TEST_DIR/main.go" <<'EOF'
package main

import (
	"fmt"
	"os"
	"os/exec"
)

func main() {
	fmt.Println("Hello, World!")
	
	// Intentional security issue: command injection vulnerability
	userInput := os.Args[1]
	cmd := exec.Command("sh", "-c", userInput) // G204
	cmd.Run()
	
	// Intentional security issue: file permissions
	os.WriteFile("test.txt", []byte("data"), 0777) // G306
}
EOF

echo "Testing gosec junit-xml output on hello world app..."
echo "Test directory: $TEST_DIR"
echo ""
echo "Created simple Go app with 2 intentional security issues:"
echo "  - G204: Command injection"
echo "  - G306: Insecure file permissions"
echo ""

for v in "${VERSIONS[@]}"; do
  echo "========================================="
  echo "Testing gosec $v"
  echo "========================================="
  
  # Build container
  echo "Building container..."
  docker build -t gosec-test:$v --quiet -<<EOF
FROM golang:1.22-alpine
RUN go install github.com/securego/gosec/v2/cmd/gosec@$v
WORKDIR /app
ENTRYPOINT ["gosec"]
EOF

  # Get version info
  echo -n "Version: "
  docker run --rm gosec-test:$v -version 2>&1 | grep "Version:" || echo "N/A"
  
  # Test 1: -out flag (should find 2 issues)
  echo ""
  echo "Test 1: -fmt=junit-xml -out=/app/output.xml"
  docker run --rm -v "$PWD/$TEST_DIR":/app gosec-test:$v \
    ./... -fmt=junit-xml -out=/app/test-$v-out.xml 2>&1 | tail -5
  
  if [ -f "$TEST_DIR/test-$v-out.xml" ]; then
    SIZE=$(wc -c < "$TEST_DIR/test-$v-out.xml")
    echo "✅ PASS: XML created ($SIZE bytes)"
    
    # Check if it's valid XML
    if head -2 "$TEST_DIR/test-$v-out.xml" | grep -q "<?xml"; then
      echo "   ✅ Valid XML header"
      
      # Check for junit structure
      if grep -q "<testsuite" "$TEST_DIR/test-$v-out.xml"; then
        echo "   ✅ JUnit structure found"
        
        # Count failures
        FAILURES=$(grep -o 'failures="[0-9]*"' "$TEST_DIR/test-$v-out.xml" | head -1 | grep -o '[0-9]*' || echo "0")
        echo "   📊 Failures reported: $FAILURES"
      else
        echo "   ⚠️  No JUnit structure"
      fi
    else
      echo "   ❌ Not valid XML"
      echo "   First line: $(head -1 "$TEST_DIR/test-$v-out.xml")"
    fi
  else
    echo "❌ FAIL: No file created with -out flag"
  fi
  
  # Test 2: -out with -no-fail
  echo ""
  echo "Test 2: -fmt=junit-xml -out=/app/output.xml -no-fail"
  EXIT_CODE=0
  docker run --rm -v "$PWD/$TEST_DIR":/app gosec-test:$v \
    ./... -fmt=junit-xml -out=/app/test-$v-nofail.xml -no-fail 2>&1 | tail -5 || EXIT_CODE=$?
  
  echo "   Exit code: $EXIT_CODE"
  if [ -f "$TEST_DIR/test-$v-nofail.xml" ]; then
    SIZE=$(wc -c < "$TEST_DIR/test-$v-nofail.xml")
    echo "✅ PASS: XML created ($SIZE bytes)"
    
    if head -2 "$TEST_DIR/test-$v-nofail.xml" | grep -q "<?xml"; then
      echo "   ✅ Valid XML"
      FAILURES=$(grep -o 'failures="[0-9]*"' "$TEST_DIR/test-$v-nofail.xml" | head -1 | grep -o '[0-9]*' || echo "0")
      echo "   📊 Failures: $FAILURES"
    fi
  else
    echo "❌ FAIL: No file created with -no-fail"
  fi
  
  # Test 3: stdout redirect
  echo ""
  echo "Test 3: -fmt=junit-xml > output.xml (stdout redirect)"
  docker run --rm -v "$PWD/$TEST_DIR":/app gosec-test:$v \
    ./... -fmt=junit-xml > "$TEST_DIR/test-$v-stdout.xml" 2>&1 || true
  
  if [ -f "$TEST_DIR/test-$v-stdout.xml" ]; then
    SIZE=$(wc -c < "$TEST_DIR/test-$v-stdout.xml")
    echo "✅ File created ($SIZE bytes)"
    
    if head -2 "$TEST_DIR/test-$v-stdout.xml" | grep -q "<?xml"; then
      echo "   ✅ Valid XML"
      FAILURES=$(grep -o 'failures="[0-9]*"' "$TEST_DIR/test-$v-stdout.xml" | head -1 | grep -o '[0-9]*' || echo "0")
      echo "   📊 Failures: $FAILURES"
    else
      echo "   ⚠️  Text output (not XML)"
      echo "   First line: $(head -1 "$TEST_DIR/test-$v-stdout.xml")"
    fi
  else
    echo "❌ FAIL: No file from stdout redirect"
  fi
  
  # Test 4: Try without any output flags (default behavior)
  echo ""
  echo "Test 4: -fmt=junit-xml (no -out, no redirect)"
  OUTPUT=$(docker run --rm -v "$PWD/$TEST_DIR":/app gosec-test:$v \
    ./... -fmt=junit-xml 2>&1 || true)
  
  if echo "$OUTPUT" | head -2 | grep -q "<?xml"; then
    echo "✅ XML output to stdout"
    echo "$OUTPUT" > "$TEST_DIR/test-$v-default.xml"
    FAILURES=$(echo "$OUTPUT" | grep -o 'failures="[0-9]*"' | head -1 | grep -o '[0-9]*' || echo "0")
    echo "   📊 Failures: $FAILURES"
  else
    echo "❌ No XML to stdout"
    echo "   Output: $(echo "$OUTPUT" | head -1)"
  fi
  
  echo ""
done

# Summary
echo "========================================="
echo "SUMMARY"
echo "========================================="
echo ""
echo "Files created in $TEST_DIR/:"
ls -lh "$TEST_DIR/" | grep -v "^total" | grep -v "go.mod" | grep -v "main.go" || echo "No test files created"
echo ""
echo "Valid XML files with JUnit structure:"
for f in "$TEST_DIR"/test-*.xml; do
  if [ -f "$f" ]; then
    if head -2 "$f" | grep -q "<?xml" && grep -q "<testsuite" "$f" 2>/dev/null; then
      FAILURES=$(grep -o 'failures="[0-9]*"' "$f" | head -1 | grep -o '[0-9]*' || echo "?")
      echo "✅ $(basename $f) - $FAILURES failures"
    elif head -2 "$f" | grep -q "<?xml"; then
      echo "⚠️  $(basename $f) - XML but no JUnit structure"
    else
      echo "❌ $(basename $f) - not valid XML"
    fi
  fi
done

echo ""
echo "Test complete. Review files in: $TEST_DIR/"
echo ""
echo "Expected: 2 security issues (G204 + G306)"
